"""
Weather data utilities for RNN sequence prediction.

This module provides functions for downloading weather data from Meteostat
and preparing it for training recurrent neural networks.

Generated by Claude Sonnet 4.5
"""

import numpy as np
import torch
from torch.utils.data import TensorDataset, DataLoader
from meteostat import Point, Hourly
from datetime import datetime


def fetch_temperature_data(latitude, longitude, start_date, end_date):
    """
    Download hourly temperature data from Meteostat.
    
    Arguments:
        latitude: Latitude of the location
        longitude: Longitude of the location
        start_date: datetime object for start date
        end_date: datetime object for end date
        
    Returns:
        pandas DataFrame with weather data
    """
    location = Point(latitude, longitude)
    data = Hourly(location, start_date, end_date)
    data = data.fetch()
    return data


def create_sequences(temp_series, seq_length=24):
    """
    Convert a temperature time series into sequences for RNN training.
    
    This creates a many-to-one prediction task where the model predicts
    the next temperature value given seq_length previous values.
    
    Arguments:
        temp_series: pandas Series or numpy array of temperature values
        seq_length: length of input sequences (default 24 hours)
    
    Returns: x, y, mean, std
        x: input sequences (num_sequences, seq_length, 1)
        y: target values (num_sequences,)
        mean, std: normalization parameters (to denormalize predictions)
    """
    # Remove NaN values
    temp_data = temp_series.dropna().values
    
    # Normalize the data (important for neural network training!)
    mean = temp_data.mean()
    std = temp_data.std()
    temp_normalized = (temp_data - mean) / std
    
    # Create sequences
    num_sequences = len(temp_normalized) - seq_length
    x = np.zeros((num_sequences, seq_length, 1))
    y = np.zeros(num_sequences)
    
    for i in range(num_sequences):
        x[i, :, 0] = temp_normalized[i:i + seq_length]
        y[i] = temp_normalized[i + seq_length]
    
    return x, y, mean, std


def create_dataloaders(x, y, batch_size=32, train_split=0.8, num_workers=2):
    """
    Create PyTorch DataLoader objects for training and validation.
    
    Arguments:
        x: input sequences numpy array
        y: target values numpy array
        batch_size: batch size for training
        train_split: fraction of data to use for training (rest is validation)
        num_workers: number of worker processes for data loading
        
    Returns:
        train_loader, val_loader: PyTorch DataLoader objects
    """
    # Convert to tensors
    tensor_x = torch.Tensor(x)
    tensor_y = torch.Tensor(np.expand_dims(y, axis=1))
    
    # Create dataset
    dataset = TensorDataset(tensor_x, tensor_y)
    
    # Split into train and validation
    train_set, val_set = torch.utils.data.random_split(
        dataset, [train_split, 1 - train_split]
    )
    
    # Create dataloaders
    train_loader = DataLoader(
        train_set, 
        batch_size=batch_size,
        shuffle=True, 
        num_workers=num_workers
    )
    val_loader = DataLoader(
        val_set, 
        batch_size=batch_size,
        shuffle=False, 
        num_workers=num_workers
    )
    
    return train_loader, val_loader


def prepare_weather_data(latitude=38.4496, longitude=-78.8689, 
                        start_year=2022, end_year=2024,
                        seq_length=24, batch_size=32):
    """
    Convenience function to download and prepare weather data in one call.
    
    Arguments:
        latitude: Latitude of the location (default: Harrisonburg, VA)
        longitude: Longitude of the location (default: Harrisonburg, VA)
        start_year: Starting year for data
        end_year: Ending year for data
        seq_length: Length of input sequences (default 24 hours)
        batch_size: Batch size for dataloaders
        
    Returns:
        data: pandas DataFrame with raw weather data
        train_loader: Training DataLoader
        val_loader: Validation DataLoader
        mean: Mean temperature (for denormalization)
        std: Std deviation of temperature (for denormalization)
    """
    # Download data
    start_date = datetime(start_year, 1, 1)
    end_date = datetime(end_year, 1, 1)
    data = fetch_temperature_data(latitude, longitude, start_date, end_date)
    
    # Determine which temperature column to use
    # Hourly data typically has 'temp', but check what's available
    if 'temp' in data.columns:
        temp_column = 'temp'
    elif 'tavg' in data.columns:
        temp_column = 'tavg'
    else:
        raise ValueError(f"No temperature column found. Available columns: {data.columns.tolist()}")
    
    # Create sequences
    x, y, mean, std = create_sequences(data[temp_column], seq_length=seq_length)
    
    # Create dataloaders
    train_loader, val_loader = create_dataloaders(x, y, batch_size=batch_size)
    
    print(f"Downloaded {len(data)} hours of weather data")
    print(f"That's approximately {len(data)/24:.0f} days")
    print(f"Created {len(x)} training sequences")
    print(f"Sequence length: {seq_length} hours")
    print(f"Temperature column: '{temp_column}'")
    print(f"Temperature mean: {mean:.2f}°C, std: {std:.2f}°C")
    
    return data, train_loader, val_loader, mean, std
