"""Utility functions for reading CSV and JSON files."""

import csv
import json
import os

# Assume that the data files are in the same directory as this file.
os.chdir(os.path.dirname(__file__))


def read_csv(filename, header=True):
    """Read a CSV file and return its contents.

    Each row is represented as a list of strings. If `header` is True,
    the first line of the file is parsed as the column names.

    Args:
        filename (str): The name of the CSV file to read.
        header (bool): Whether the first line has column names.

    Returns:
        tuple: Column names (list[str]) and CSV data (list[list[str]]).
    """
    with open(filename, newline="") as file:
        reader = csv.reader(file)
        if header:
            names = next(reader)
        data = [row for row in reader]
        if not header:
            names = [""] * len(data[0])
        return names, data


def read_json(filename):
    """Read a JSON file and return its contents.

    The returned value can be a dictionary, list, string, number,
    boolean, or None, depending on the structure of the JSON file.

    Args:
        filename (str): The name of the JSON file to read.

    Returns:
        Any: The JSON data parsed as a Python object.
    """
    with open(filename) as file:
        return json.load(file)
