/*
    turtle.c

    Simple array-based turtle graphics engine in C. Exports to BMP files.

    Author: Mike Lam, James Madison University, August 2015

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

    Known issues:

        - "filled" polygons are not always entirely filled, especially when
          side angles are very acute; there is probably an incorrect floating-
          point or integer conversion somewhere

*/

#include "turtle.h"

#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <math.h>


/**  DEFINITIONS  **/

#define ABS(X) ((X)>0 ? (X) : (-(X)))

#define PI 3.141592653589793

#define MAX_POLYGON_VERTICES 128

// pixel data (red, green, blue triplet)
typedef struct {
    char red;
    char green;
    char blue;
} rgb_t;


/**  GLOBAL TURTLE STATE  **/

typedef struct {
    double  xpos;       // current position and heading
    double  ypos;       // (uses floating-point numbers for
    double  heading;    //  increased accuracy)

    rgb_t  pen_color;   // current pen color
    rgb_t  fill_color;  // current fill color
    bool   pendown;     // currently drawing?
    bool   filled;      // currently filling?
} turtle_t;

turtle_t main_turtle; 
turtle_t backup_turtle;

rgb_t *main_turtle_image = NULL;        // 2d pixel data field

int    main_field_width  = 0;           // size in pixels
int    main_field_height = 0;

bool   main_field_save_frames = false;  // currently saving video frames?
int    main_field_frame_count    = 0;   // current video frame counter
int    main_field_frame_interval = 10;  // pixels per frame
int    main_field_pixel_count    = 0;   // total pixels drawn by turtle since
                                        // beginning of video
int    main_turtle_poly_vertex_count = 0;       // polygon vertex count
double main_turtle_polyX[MAX_POLYGON_VERTICES]; // polygon vertex x-coords
double main_turtle_polyY[MAX_POLYGON_VERTICES]; // polygon vertex y-coords


/**  TURTLE FUNCTIONS  **/

void turtle_init(int width, int height)
{
    int total_size = sizeof(rgb_t) * width * height;

    // free previous image array if necessary
    if (main_turtle_image != NULL) {
        free(main_turtle_image);
        main_turtle_image = NULL;
    }

    // allocate new image and initialize it to white
    main_turtle_image = (rgb_t*)malloc(total_size);
    if (main_turtle_image == NULL) {
        fprintf(stderr, "Can't allocate memory for turtle image.\n");
        exit(EXIT_FAILURE);
    }
    memset(main_turtle_image, 255, total_size);

    // save field size for later
    main_field_width = width;
    main_field_height = height;

    // disable video
    main_field_save_frames = false;

    // reset turtle position and color
    turtle_reset();
}

void turtle_reset()
{
    // move turtle to middle of the field
    main_turtle.xpos = 0.0;
    main_turtle.ypos = 0.0;

    // orient to the right (0 deg)
    main_turtle.heading = 0.0;

    // default draw color is black
    main_turtle.pen_color.red = 0;
    main_turtle.pen_color.green = 0;
    main_turtle.pen_color.blue = 0;

    // default fill color is black
    main_turtle.fill_color.red = 0;
    main_turtle.fill_color.green = 255;
    main_turtle.fill_color.blue = 0;

    // default pen position is down
    main_turtle.pendown = true;

    // default fill status is off
    main_turtle.filled = false;
    main_turtle_poly_vertex_count = 0;
}

void turtle_backup() {
    backup_turtle = main_turtle;
}

void turtle_restore() {
    main_turtle = backup_turtle;
}

void turtle_forward(int pixels)
{
    // calculate (x,y) movement vector from heading
    double radians = main_turtle.heading * PI / 180.0;
    double dx = cos(radians) * pixels;
    double dy = sin(radians) * pixels;

    // delegate to another method to actually move
    turtle_goto_real(main_turtle.xpos + dx, main_turtle.ypos + dy);
}

void turtle_backward(int pixels)
{
    // opposite of "forward"
    turtle_forward(-pixels);
}

void turtle_strafe_left(int pixels) {
	turtle_turn_left(90);
	turtle_forward(pixels);
	turtle_turn_right(90);
}

void turtle_strafe_right(int pixels) {
	turtle_turn_right(90);
	turtle_forward(pixels);
	turtle_turn_left(90);
}

void turtle_turn_left(double angle)
{
    // rotate turtle heading
    main_turtle.heading += angle;

    // constrain heading to range: [0.0, 360.0)
    if (main_turtle.heading < 0.0) {
        main_turtle.heading += 360.0;
    } else if (main_turtle.heading >= 360.0) {
        main_turtle.heading -= 360.0;
    }
}

void turtle_turn_right(double angle)
{
    // opposite of "turn left"
    turtle_turn_left(-angle);
}

void turtle_pen_up()
{
    main_turtle.pendown = false;
}

void turtle_pen_down()
{
    main_turtle.pendown = true;
}

void turtle_begin_fill()
{
    main_turtle.filled = true;
    main_turtle_poly_vertex_count = 0;
}

void turtle_end_fill()
{
    // based on public-domain fill algorithm in C by Darel Rex Finley, 2007
    //   from http://alienryderflex.com/polygon_fill/

    double nodeX[MAX_POLYGON_VERTICES];     // x-coords of polygon intercepts
    int nodes;                              // size of nodeX
    int x, y, i, j;                         // current pixel and loop indices
    double temp;                            // temporary variable for sorting

    //  loop through the rows of the image
    for (y = -(main_field_height/2); y < main_field_height/2; y++) {

        //  build a list of polygon intercepts on the current line
        nodes = 0;
        j = main_turtle_poly_vertex_count-1;
        for (i = 0; i < main_turtle_poly_vertex_count; i++) {
            if ((main_turtle_polyY[i] <  (double)y &&
                 main_turtle_polyY[j] >= (double)y) ||
                (main_turtle_polyY[j] <  (double)y &&
                 main_turtle_polyY[i] >= (double)y)) {

                // intercept found; record it
                nodeX[nodes++] = (main_turtle_polyX[i] +
                        ((double)y - main_turtle_polyY[i]) /
                        (main_turtle_polyY[j] - main_turtle_polyY[i]) *
                        (main_turtle_polyX[j] - main_turtle_polyX[i]));
            }
            j = i;
            if (nodes >= MAX_POLYGON_VERTICES) {
                fprintf(stderr, "Too many intercepts in fill algorithm!\n");
                exit(EXIT_FAILURE);
            }
        }

        //  sort the nodes via simple insertion sort
        for (i = 1; i < nodes; i++) {
            temp = nodeX[i];
            for (j = i; j > 0 && temp < nodeX[j-1]; j--) {
                nodeX[j] = nodeX[j-1];
            }
            nodeX[j] = temp;
        }

        //  fill the pixels between node pairs
        for (i = 0; i < nodes; i += 2) {
            for (x = (int)floor(nodeX[i])+1; x < (int)ceil(nodeX[i+1]); x++) {
                turtle_fill_pixel(x, y);
            }
        }
    }

    main_turtle.filled = false;

    // redraw polygon (filling is imperfect and can occasionally occlude sides)
    for (i = 0; i < main_turtle_poly_vertex_count; i++) {
        int x0 = (int)round(main_turtle_polyX[i]);
        int y0 = (int)round(main_turtle_polyY[i]);
        int x1 = (int)round(main_turtle_polyX[(i+1) %
            main_turtle_poly_vertex_count]);
        int y1 = (int)round(main_turtle_polyY[(i+1) %
            main_turtle_poly_vertex_count]);
        turtle_draw_line(x0, y0, x1, y1);
    }
}

void turtle_goto(int x, int y)
{
    turtle_goto_real((double)x, (double)y);
}

void turtle_goto_real(double x, double y)
{
    // draw line if pen is down
    if (main_turtle.pendown) {
        turtle_draw_line((int)round(main_turtle.xpos),
                         (int)round(main_turtle.ypos),
                         (int)round(x),
                         (int)round(y));
    }

    // change current turtle position
    main_turtle.xpos = (double)x;
    main_turtle.ypos = (double)y;

    // track coordinates for filling
    if (main_turtle.filled && main_turtle.pendown &&
            main_turtle_poly_vertex_count < MAX_POLYGON_VERTICES) {
        main_turtle_polyX[main_turtle_poly_vertex_count] = x;
        main_turtle_polyY[main_turtle_poly_vertex_count] = y;
        main_turtle_poly_vertex_count++;
    }
}

void turtle_set_heading(double angle)
{
    main_turtle.heading = angle;
}

void turtle_set_pen_color(int red, int green, int blue)
{
    main_turtle.pen_color.red = red;
    main_turtle.pen_color.green = green;
    main_turtle.pen_color.blue = blue;
}

void turtle_set_fill_color(int red, int green, int blue)
{
    main_turtle.fill_color.red = red;
    main_turtle.fill_color.green = green;
    main_turtle.fill_color.blue = blue;
}

void turtle_dot()
{
    // draw a pixel at the current location, regardless of pen status
    turtle_draw_pixel((int)round(main_turtle.xpos),
                      (int)round(main_turtle.ypos));
}

void turtle_draw_pixel(int x, int y)
{
    // calculate pixel offset in image data array
    int idx = main_field_width * (y+main_field_height/2)
                               + (x+main_field_width/2);

    // check to make sure it's not out of bounds
    if (idx < 0 || idx > main_field_width*main_field_height) {
        fprintf(stderr, "Pixel out of bounds: (%d,%d)\n", x, y);
    } else {
        main_turtle_image[idx].red   = main_turtle.pen_color.red;
        main_turtle_image[idx].green = main_turtle.pen_color.green;
        main_turtle_image[idx].blue  = main_turtle.pen_color.blue;
    }

    // track total pixels drawn and emit video frame if a frame interval has
    // been crossed (and only if video saving is enabled, of course)
    if (main_field_save_frames &&
            main_field_pixel_count++ % main_field_frame_interval == 0) {
        turtle_save_frame();
    }
}

void turtle_fill_pixel(int x, int y)
{
    // calculate pixel offset in image data array
    int idx = main_field_width * (y+main_field_height/2)
                               + (x+main_field_width/2);

    // check to make sure it's not out of bounds
    if (idx >= 0 && idx < main_field_width*main_field_height) {
        main_turtle_image[idx].red   = main_turtle.fill_color.red;
        main_turtle_image[idx].green = main_turtle.fill_color.green;
        main_turtle_image[idx].blue  = main_turtle.fill_color.blue;
    }
}

void turtle_draw_line(int x0, int y0, int x1, int y1)
{
    // uses a variant of Bresenham's line algorithm:
    //   https://en.wikipedia.org/wiki/Talk:Bresenham%27s_line_algorithm

    int absX = ABS(x1-x0);          // absolute value of coordinate distances
    int absY = ABS(y1-y0);
    int offX = x0<x1 ? 1 : -1;      // line-drawing direction offsets
    int offY = y0<y1 ? 1 : -1;
    int x = x0;                     // incremental location
    int y = y0;
    int err;

    turtle_draw_pixel(x, y);
    if (absX > absY) {

        // line is more horizontal; increment along x-axis
        err = absX / 2;
        while (x != x1) {
            err = err - absY;
            if (err < 0) {
                y   += offY;
                err += absX;
            }
            x += offX;
            turtle_draw_pixel(x,y);
        }
    } else {

        // line is more vertical; increment along y-axis
        err = absY / 2;
        while (y != y1) {
            err = err - absX;
            if (err < 0) {
                x   += offX;
                err += absY;
            }
            y += offY;
            turtle_draw_pixel(x,y);
        }
    }
}

void turtle_draw_circle(int x0, int y0, int radius)
{
    // implementation based on midpoint circle algorithm:
    //   https://en.wikipedia.org/wiki/Midpoint_circle_algorithm

    int x = radius;
    int y = 0;
    int switch_criteria = 1 - x;

    // naive attempt at fill algorithm; leaves gaps
    /*
     *if (main_turtle.filled) {
     *    rgb_t saved_pen_color = main_turtle.pen_color;
     *    main_turtle.pen_color = main_turtle.fill_color;
     *    main_turtle.filled = false;
     *    for (int r = 0; r < radius; r++) {
     *        turtle_draw_circle(x0, y0, r);
     *    }
     *    main_turtle.filled = true;
     *    main_turtle.pen_color = saved_pen_color;
     *}
     */

    while (x >= y) {
        turtle_draw_pixel( x + x0,  y + y0);
        turtle_draw_pixel( y + x0,  x + y0);
        turtle_draw_pixel(-x + x0,  y + y0);
        turtle_draw_pixel(-y + x0,  x + y0);
        turtle_draw_pixel(-x + x0, -y + y0);
        turtle_draw_pixel(-y + x0, -x + y0);
        turtle_draw_pixel( x + x0, -y + y0);
        turtle_draw_pixel( y + x0, -x + y0);
        y++;
        if (switch_criteria <= 0) {
            switch_criteria += 2 * y + 1;       // no x-coordinate change
        } else {
            x--;
            switch_criteria += 2 * (y - x) + 1;
        }
    }
}

/*
    Fill a circle at the given coordinates with the given radius, regardless of
    current turtle location or pen status.
*/
void turtle_fill_circle(int x0, int y0, int radius) {
	
	int rad_sq = radius * radius;
	
	// Naive algorithm, pretty ugly due to no antialiasing:
	for (int x = x0 - radius; x < x0 + radius; x++) {
		for (int y = y0 - radius; y < y0 + radius; y++) {
			int dx = x - x0;
			int dy = y - y0;
			int dsq = (dx * dx) + (dy * dy);
			if (dsq < rad_sq) turtle_fill_pixel(x, y);
		}
	}
}

void turtle_fill_circle_here(int radius) 
{
    turtle_fill_circle(main_turtle.xpos, main_turtle.ypos, radius);
}

/*
    Draw a turtle at the current pen location.
*/
void turtle_draw_turtle() 
{
        
    // We are going to make our own backup of the turtle, since turtle_backup()
    // only gives us one level of undo.
    turtle_t original_turtle = main_turtle;
    
    turtle_pen_up();
    
    // Draw the legs:
    for (int i = -1; i < 2; i+=2) {
        for (int j = -1; j < 2; j+=2) {
            turtle_backup();
                turtle_forward(i * 7);
                turtle_strafe_left(j * 7);
            
                turtle_set_fill_color(
                    main_turtle.pen_color.red, 
                    main_turtle.pen_color.green, 
                    main_turtle.pen_color.blue
                );
                turtle_fill_circle_here(5);
            
                turtle_set_fill_color(
                    original_turtle.fill_color.red,
                    original_turtle.fill_color.green,
                    original_turtle.fill_color.blue
                );
                turtle_fill_circle_here(3);
            turtle_restore();
        }
    }

    // Draw the head:
    turtle_backup();
        turtle_forward(10);
        turtle_set_fill_color(
            main_turtle.pen_color.red, 
            main_turtle.pen_color.green, 
            main_turtle.pen_color.blue
        );
        turtle_fill_circle_here(5);
    
        turtle_set_fill_color(
            original_turtle.fill_color.red,
            original_turtle.fill_color.green,
            original_turtle.fill_color.blue
        );
        turtle_fill_circle_here(3);
    turtle_restore();

    // Draw the body
    for (int i = 9; i >= 0; i-=4) {
        turtle_backup();
            turtle_set_fill_color(
                main_turtle.pen_color.red, 
                main_turtle.pen_color.green, 
                main_turtle.pen_color.blue
            );
            turtle_fill_circle_here(i+2);
    
            turtle_set_fill_color(
                original_turtle.fill_color.red,
                original_turtle.fill_color.green,
                original_turtle.fill_color.blue
            );
            turtle_fill_circle_here(i);
        turtle_restore();
    }
            
    // Restore the original turtle position:
    main_turtle = original_turtle;
}

// /*
//     Draw a turtle at the current pen location.
// */
// void turtle_draw_turtle() {
//
//     // Save the original location, pen info, etc:
//
//     bool orig_pen_down = main_turtle.pendown;
//     double orig_x = main_turtle.xpos;
//     double orig_y = main_turtle.ypos;
//     rgb_t orig_fill_color = main_turtle.fill_color;
//     rgb_t orig_pen_color = main_turtle.pen_color;
//
//     turtle_pen_up();
//
//     // Draw the legs:
//
//     turtle_set_fill_color(0, 0, 0);
//     turtle_forward(14);
//     turtle_strafe_left(14);
//     turtle_fill_circle(main_turtle.xpos, main_turtle.ypos, 7);
//     turtle_strafe_right(14);
//     turtle_backward(14);
//
//     turtle_set_fill_color(orig_fill_color.red, orig_fill_color.green, orig_fill_color.blue);
//     turtle_forward(14);
//     turtle_strafe_left(14);
//     turtle_fill_circle(main_turtle.xpos, main_turtle.ypos, 5);
//     turtle_strafe_right(14);
//     turtle_backward(14);
//
//
//     turtle_set_fill_color(0, 0, 0);
//     turtle_forward(14);
//     turtle_strafe_right(14);
//     turtle_fill_circle(main_turtle.xpos, main_turtle.ypos, 7);
//     turtle_strafe_left(14);
//     turtle_backward(14);
//
//     turtle_set_fill_color(orig_fill_color.red, orig_fill_color.green, orig_fill_color.blue);
//     turtle_forward(14);
//     turtle_strafe_right(14);
//     turtle_fill_circle(main_turtle.xpos, main_turtle.ypos, 5);
//     turtle_strafe_left(14);
//     turtle_backward(14);
//
//
//     turtle_set_fill_color(0, 0, 0);
//     turtle_backward(14);
//     turtle_strafe_left(14);
//     turtle_fill_circle(main_turtle.xpos, main_turtle.ypos, 7);
//     turtle_strafe_right(14);
//     turtle_forward(14);
//
//     turtle_set_fill_color(orig_fill_color.red, orig_fill_color.green, orig_fill_color.blue);
//     turtle_backward(14);
//     turtle_strafe_left(14);
//     turtle_fill_circle(main_turtle.xpos, main_turtle.ypos, 5);
//     turtle_strafe_right(14);
//     turtle_forward(14);
//
//
//     turtle_set_fill_color(0, 0, 0);
//     turtle_backward(14);
//     turtle_strafe_right(14);
//     turtle_fill_circle(main_turtle.xpos, main_turtle.ypos, 7);
//     turtle_strafe_left(14);
//     turtle_forward(14);
//
//     turtle_set_fill_color(orig_fill_color.red, orig_fill_color.green, orig_fill_color.blue);
//     turtle_backward(14);
//     turtle_strafe_right(14);
//     turtle_fill_circle(main_turtle.xpos, main_turtle.ypos, 5);
//     turtle_strafe_left(14);
//     turtle_forward(14);
//
//     // Draw the tail
//     turtle_strafe_left(1);
//     turtle_set_pen_color(0,0,0);
//
//     turtle_pen_down();
//         turtle_backward(24);
//         turtle_strafe_left(1);
//         turtle_forward(24);
//     turtle_pen_up();
//
//     turtle_set_pen_color(orig_pen_color.red, orig_pen_color.green, orig_pen_color.blue);
//     turtle_strafe_right(2);
//
//     turtle_strafe_right(1);
//     turtle_set_pen_color(0,0,0);
//
//     turtle_pen_down();
//         turtle_backward(24);
//         turtle_strafe_right(1);
//         turtle_forward(24);
//     turtle_pen_up();
//
//     turtle_set_pen_color(orig_pen_color.red, orig_pen_color.green, orig_pen_color.blue);
//     turtle_strafe_left(2);
//
//     turtle_set_pen_color(orig_fill_color.red, orig_fill_color.green, orig_fill_color.blue);
//     turtle_pen_down();
//     turtle_backward(22);
//     turtle_pen_up();
//     turtle_forward(22);
//     turtle_set_pen_color(orig_pen_color.red, orig_pen_color.green, orig_pen_color.blue);
//
//     // Draw the head:
//
//     turtle_forward(20);
//
//     turtle_set_fill_color(0, 0, 0);
//     turtle_fill_circle(main_turtle.xpos, main_turtle.ypos, 7);
//
//     turtle_set_fill_color(orig_fill_color.red, orig_fill_color.green, orig_fill_color.blue);
//     turtle_fill_circle(main_turtle.xpos, main_turtle.ypos, 5);
//
//     turtle_backward(20);
//
//     // Draw the body:
//
//     turtle_set_fill_color(0, 0, 0);
//     turtle_fill_circle(main_turtle.xpos, main_turtle.ypos, 18);
//
//     turtle_set_fill_color(orig_fill_color.red, orig_fill_color.green, orig_fill_color.blue);
//     turtle_fill_circle(main_turtle.xpos, main_turtle.ypos, 16);
//
//     turtle_set_fill_color(0, 0, 0);
//     turtle_fill_circle(main_turtle.xpos, main_turtle.ypos, 14);
//
//     turtle_set_fill_color(orig_fill_color.red, orig_fill_color.green, orig_fill_color.blue);
//     turtle_fill_circle(main_turtle.xpos, main_turtle.ypos, 12);
//
//     turtle_set_fill_color(0, 0, 0);
//     turtle_fill_circle(main_turtle.xpos, main_turtle.ypos, 10);
//
//     turtle_set_fill_color(orig_fill_color.red, orig_fill_color.green, orig_fill_color.blue);
//     turtle_fill_circle(main_turtle.xpos, main_turtle.ypos, 8);
//
//     // Restore original settings:
//
//     main_turtle.pendown = orig_pen_down;
//     main_turtle.xpos = orig_x;
//     main_turtle.ypos = orig_y;
//
//     main_turtle.fill_color = orig_fill_color;
//     main_turtle.pen_color = orig_pen_color;
// }

void turtle_begin_video(int pixels_per_frame)
{
    main_field_save_frames = true;
    main_field_frame_count = 0;
    main_field_frame_interval = pixels_per_frame;
    main_field_pixel_count = 0;
}

void turtle_save_frame()
{
    char filename[32];
    sprintf(filename, "frame%05d.bmp", ++main_field_frame_count);
    turtle_save_bmp(filename);
}

void turtle_end_video()
{
    main_field_save_frames = false;
}

void turtle_cleanup()
{
    // free image array if allocated
    if (main_turtle_image != NULL) {
        free(main_turtle_image);
        main_turtle_image = NULL;
    }
}


// the rest of this file is based on GPL'ed code from:
// http://cpansearch.perl.org/src/DHUNT/PDL-Planet-0.12/libimage/bmp.c

struct BMPHeader
{
    char bfType[2];       // "BM"
    int bfSize;           // size of file in bytes
    int bfReserved;       // set to 0
    int bfOffBits;        // byte offset to actual bitmap data (= 54)
    int biSize;           // size of BITMAPINFOHEADER, in bytes (= 40)
    int biWidth;          // width of image, in pixels
    int biHeight;         // height of images, in pixels
    short biPlanes;       // number of planes in target device (set to 1)
    short biBitCount;     // bits per pixel (24 in this case)
    int biCompression;    // type of compression (0 if no compression)
    int biSizeImage;      // image size, in bytes (0 if no compression)
    int biXPelsPerMeter;  // resolution in pixels/meter of display device
    int biYPelsPerMeter;  // resolution in pixels/meter of display device
    int biClrUsed;        // number of colors in the color table (if 0, use
                          // maximum allowed by biBitCount)
    int biClrImportant;   // number of important colors.  If 0, all colors
                          // are important
};

void turtle_save_bmp(const char *filename)
{
    int i, j, ipos;
    int bytesPerLine;
    unsigned char *line;
    FILE *file;
    struct BMPHeader bmph;
    int width = main_field_width;
    int height = main_field_height;
    char *rgb = (char*)main_turtle_image;

    // the length of each line must be a multiple of 4 bytes
    bytesPerLine = (3 * (width + 1) / 4) * 4;

    strcpy(bmph.bfType, "BM");
    bmph.bfOffBits = 54;
    bmph.bfSize = bmph.bfOffBits + bytesPerLine * height;
    bmph.bfReserved = 0;
    bmph.biSize = 40;
    bmph.biWidth = width;
    bmph.biHeight = height;
    bmph.biPlanes = 1;
    bmph.biBitCount = 24;
    bmph.biCompression = 0;
    bmph.biSizeImage = bytesPerLine * height;
    bmph.biXPelsPerMeter = 0;
    bmph.biYPelsPerMeter = 0;
    bmph.biClrUsed = 0;
    bmph.biClrImportant = 0;

    file = fopen (filename, "wb");
    if (file == NULL) {
        fprintf(stderr, "Could not write to file: %s\n", filename);
        exit(EXIT_FAILURE);
    }

    fwrite(&bmph.bfType, 2, 1, file);
    fwrite(&bmph.bfSize, 4, 1, file);
    fwrite(&bmph.bfReserved, 4, 1, file);
    fwrite(&bmph.bfOffBits, 4, 1, file);
    fwrite(&bmph.biSize, 4, 1, file);
    fwrite(&bmph.biWidth, 4, 1, file);
    fwrite(&bmph.biHeight, 4, 1, file);
    fwrite(&bmph.biPlanes, 2, 1, file);
    fwrite(&bmph.biBitCount, 2, 1, file);
    fwrite(&bmph.biCompression, 4, 1, file);
    fwrite(&bmph.biSizeImage, 4, 1, file);
    fwrite(&bmph.biXPelsPerMeter, 4, 1, file);
    fwrite(&bmph.biYPelsPerMeter, 4, 1, file);
    fwrite(&bmph.biClrUsed, 4, 1, file);
    fwrite(&bmph.biClrImportant, 4, 1, file);

    line = (unsigned char*)malloc(bytesPerLine);
    if (line == NULL) {
        fprintf(stderr, "Can't allocate memory for BMP file.\n");
        exit(EXIT_FAILURE);
    }

    for (i = 0; i < height; i++) {
        for (j = 0; j < width; j++) {
            ipos = 3 * (width * i + j);
            line[3*j] = rgb[ipos + 2];
            line[3*j+1] = rgb[ipos + 1];
            line[3*j+2] = rgb[ipos];
        }
        fwrite(line, bytesPerLine, 1, file);
    }

    free(line);
    fclose(file);
}

